import Footer from '@/components/footer';
import Navbar from '@/components/navbar';
import Partnerships from '@/components/partnerships';
import ProductCard from '@/components/product-card';
import { Input } from '@/components/ui/input';
import { Partnership } from '@/types/partnerships';
import { ProductPagination, Tumbler } from '@/types/products';
import { router } from '@inertiajs/react';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import ArrowForwardIcon from '@mui/icons-material/ArrowForward';
import Pagination from '@mui/material/Pagination';
import PaginationItem from '@mui/material/PaginationItem';
import Stack from '@mui/material/Stack';
import AOS from 'aos';
import 'aos/dist/aos.css';
import { Search } from 'lucide-react';
import { useEffect, useState } from 'react';

interface AllProductsProps {
    products: ProductPagination;
    partnerships: Partnership[];
    tumbler: Tumbler;
    query?: string;
}

export default function AllProducts({
    products,
    partnerships,
    tumbler,
    query: initialQuery = '',
}: AllProductsProps) {
    const [query, setQuery] = useState(initialQuery);

    const handleSearch = (e: React.FormEvent) => {
        e.preventDefault();
        router.get('/products', { search: query }, { preserveScroll: true });
    };

    const handlePageChange = (_: React.ChangeEvent<unknown>, value: number) => {
        router.get(`/products?page=${value}`, {}, { preserveScroll: true });
    };

    useEffect(() => {
        AOS.init({ duration: 900, once: true });
    }, []);

    return (
        <>
            <div className="flex min-h-screen flex-col">
                <Navbar />
                <main className="flex-1">
                    <section className="mx-5 mt-2 mb-7 grid items-start gap-2 md:mx-15 md:grid-cols-[auto_1fr] md:gap-3 md:gap-6">
                        <div>
                            <h2 className="text-2xl font-bold md:text-5xl">
                                <span className="text-[#8BEBA2]">Produk </span>
                                <span className="text-[#06A3C7]">
                                    Merchandise
                                </span>
                            </h2>
                            <p className="text-sm font-semibold md:text-lg">
                                Our best seller
                            </p>
                        </div>
                        <form
                            onSubmit={handleSearch}
                            className="relative focus:outline-none"
                        >
                            <Input
                                type="text"
                                value={query}
                                onChange={(e) => setQuery(e.target.value)}
                                placeholder="Cari produk..."
                                className="h-8 w-full rounded-full border-none bg-[#C5E9F2] pl-3 text-[10px] font-medium text-black focus-visible:ring-[2px] focus-visible:ring-[#06A3C7] md:h-12 md:pl-5 md:text-sm dark:bg-[#DDDDDD] dark:placeholder:text-black"
                            />
                            <button className="absolute top-1/2 right-4 -translate-y-1/2 cursor-pointer">
                                <Search className="h-4 w-4 text-black md:h-6 md:w-6" />
                            </button>
                        </form>
                    </section>
                    <section className="mx-5 mb-7 md:mx-15 md:mb-10">
                        <div className="grid grid-cols-2 gap-3 md:grid-cols-3 md:gap-10">
                            {products.data.length > 0 ? (
                                products.data.map((product) => (
                                    <ProductCard product={product} />
                                ))
                            ) : (
                                <p className="col-span-3 text-center text-xs text-[#969696] md:text-base">
                                    Produk tidak ditemukan.
                                </p>
                            )}
                        </div>
                        <div className="my-5 flex justify-center md:my-6">
                            <Stack spacing={2}>
                                <Pagination
                                    count={products.last_page}
                                    page={products.current_page}
                                    onChange={handlePageChange}
                                    renderItem={(item) => (
                                        <PaginationItem
                                            slots={{
                                                previous: ArrowBackIcon,
                                                next: ArrowForwardIcon,
                                            }}
                                            {...item}
                                            sx={{
                                                fontSize: {
                                                    xs: '0.75rem',
                                                    sm: '0.875rem',
                                                    md: '1rem',
                                                },
                                            }}
                                            className={`rounded-full transition-all ${
                                                item.selected
                                                    ? 'bg-gradient-to-r from-[#7DD0E2] to-[#C7FBC4] font-semibold text-black'
                                                    : 'hover:bg-[#DDDDDD]'
                                            } `}
                                        />
                                    )}
                                />
                            </Stack>
                        </div>
                        <div className="flex flex-col items-center justify-between gap-2 rounded-xl bg-gradient-to-r from-[#06A3C7] to-[#035061] md:flex-row md:gap-0 md:rounded-2xl">
                            <img
                                src={tumbler.main_image}
                                className="mt-3 w-50 md:mt-0 md:w-90 md:py-5 md:pl-5"
                            />
                            <div className="mb-3 px-3 text-center text-white md:mb-0 md:px-0 md:py-5 md:pr-5 md:text-right">
                                <p className="text-lg font-bold md:text-4xl">
                                    Tumbler Kostum
                                </p>
                                <p className="text-xs md:text-lg">
                                    Buat followers penasaran buat brand kamu
                                    makin viral!
                                </p>
                            </div>
                        </div>
                    </section>
                </main>
                <Partnerships data={partnerships} />
                <Footer />
            </div>
        </>
    );
}
