import ArticleCard from '@/components/article-card';
import Footer from '@/components/footer';
import Navbar from '@/components/navbar';
import Partnerships from '@/components/partnerships';
import { Input } from '@/components/ui/input';
import { Article } from '@/types/articles';
import { Partnership } from '@/types/partnerships';
import { router } from '@inertiajs/react';
import AOS from 'aos';
import 'aos/dist/aos.css';
import { CalendarDays, Search, UserRound } from 'lucide-react';
import { useEffect, useState } from 'react';

interface ArticlesProps {
    featuredArticle: Article;
    sideArticles: Article[];
    partnerships: Partnership[];
    query?: string;
}

export default function Articles({
    featuredArticle,
    sideArticles,
    partnerships,
    query: initialQuery = '',
}: ArticlesProps) {
    const [query, setQuery] = useState(initialQuery);

    const handleSearch = (e: React.FormEvent) => {
        e.preventDefault();
        router.get(
            `/articles/all`,
            { search: query },
            { preserveScroll: true },
        );
    };

    useEffect(() => {
        AOS.init({ duration: 900, once: true });
    });

    return (
        <>
            <div className="flex min-h-screen flex-col">
                <Navbar />
                <main className="flex-1">
                    <section className="mx-5 mt-2 mb-7 grid items-start gap-2 md:mx-15 md:grid-cols-[auto_1fr] md:gap-6">
                        <div>
                            <h2 className="text-2xl font-bold text-[#06A3C7] md:text-5xl">
                                Artikel
                            </h2>
                            <p className="text-sm font-semibold md:text-lg">
                                Baca hari ini, hebat esok hari
                            </p>
                        </div>
                        <form
                            onSubmit={handleSearch}
                            className="relative focus:outline-none"
                        >
                            <Input
                                type="text"
                                value={query}
                                onChange={(e) => setQuery(e.target.value)}
                                placeholder="Cari artikel..."
                                className="h-8 w-full rounded-full border-none bg-[#C5E9F2] pl-3 text-[10px] font-medium text-black focus-visible:ring-[2px] focus-visible:ring-[#06A3C7] md:h-12 md:pl-5 md:text-sm dark:bg-[#DDDDDD] dark:placeholder:text-black"
                            />
                            <button className="absolute top-1/2 right-4 -translate-y-1/2 cursor-pointer">
                                <Search className="h-4 w-4 text-black md:h-6 md:w-6" />
                            </button>
                        </form>
                    </section>
                    <section className="mx-5 mb-7 md:mx-15 md:mb-10">
                        <div className="flex flex-col gap-5 md:flex-row">
                            <div
                                className="grid flex-[2] gap-1 md:gap-3"
                                data-aos="zoom-in"
                            >
                                <div className="relative h-45 w-full overflow-hidden rounded-lg md:h-102 md:rounded-2xl">
                                    <img
                                        src={featuredArticle.image_url}
                                        className="h-full w-full object-cover transition-transform"
                                    />
                                </div>
                                <div className="flex items-center gap-8">
                                    <div className="flex items-center gap-1">
                                        <CalendarDays className="h-3 w-3 text-[#06A3C7] md:h-5 md:w-5" />
                                        <p className="pt-1 text-[10px] font-medium md:text-sm">
                                            {new Date(
                                                featuredArticle.created_at,
                                            ).toLocaleDateString('id-ID', {
                                                day: 'numeric',
                                                month: 'long',
                                                year: 'numeric',
                                            })}
                                        </p>
                                    </div>
                                    <div className="flex items-center gap-1">
                                        <UserRound className="h-3 w-3 text-[#06A3C7] md:h-5 md:w-5" />
                                        <p className="pt-1 text-[10px] font-medium md:text-sm">
                                            Admin
                                        </p>
                                    </div>
                                </div>
                                <a
                                    href={`/articles/${featuredArticle.slug}`}
                                    className="text-lg/6 font-bold text-[#06A3C7] hover:underline hover:decoration-[#06A3C7] hover:underline-offset-[3px] md:text-3xl"
                                >
                                    {featuredArticle.title}
                                </a>
                            </div>
                            <div
                                className="flex flex-[1] flex-col gap-4"
                                data-aos="zoom-in"
                                data-aos-delay="300"
                            >
                                <div className="flex flex-col gap-3">
                                    {sideArticles.length > 0 ? (
                                        sideArticles.map((article) => (
                                            <ArticleCard article={article} />
                                        ))
                                    ) : (
                                        <p>Tidak ada artikel lainnya.</p>
                                    )}
                                </div>
                                <div className="flex items-center justify-center">
                                    <a
                                        href="/articles/all"
                                        className="w-full cursor-pointer rounded-full border-2 border-transparent bg-[#06A3C7] py-1 text-center text-xs font-medium text-white hover:border-solid hover:border-[#06A3C7] hover:bg-white hover:text-[#06A3C7] md:text-base dark:hover:bg-black"
                                    >
                                        Lihat semua artikel
                                    </a>
                                </div>
                            </div>
                        </div>
                    </section>
                </main>
                <Partnerships data={partnerships} />
                <Footer />
            </div>
        </>
    );
}
