import Footer from '@/components/footer';
import Navbar from '@/components/navbar';
import Partnerships from '@/components/partnerships';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { ArticlePagination } from '@/types/articles';
import { Partnership } from '@/types/partnerships';
import { router } from '@inertiajs/react';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import ArrowForwardIcon from '@mui/icons-material/ArrowForward';
import Pagination from '@mui/material/Pagination';
import PaginationItem from '@mui/material/PaginationItem';
import Stack from '@mui/material/Stack';
import AOS from 'aos';
import 'aos/dist/aos.css';
import { CalendarDays, Search, UserRound } from 'lucide-react';
import { useEffect, useState } from 'react';

interface AllArticlesProps {
    articles: ArticlePagination;
    partnerships: Partnership[];
    query?: string;
}

export default function AllArticles({
    articles,
    partnerships,
    query: initialQuery = '',
}: AllArticlesProps) {
    const [query, setQuery] = useState(initialQuery);

    const handleSearch = (e: React.FormEvent) => {
        e.preventDefault();
        router.get(
            `/articles/all`,
            { search: query },
            { preserveScroll: true },
        );
    };

    const handlePageChange = (_: React.ChangeEvent<unknown>, value: number) => {
        router.get(`/articles/all?page=${value}`, {}, { preserveScroll: true });
    };

    const isMobile = typeof window !== 'undefined' && window.innerWidth < 768;
    const maxLength = isMobile ? 50 : 100;

    useEffect(() => {
        AOS.init({ duration: 900, once: true });
    }, []);

    return (
        <>
            <div className="flex min-h-screen flex-col">
                <Navbar />
                <main className="flex-1">
                    <section className="mx-5 mt-2 mb-7 grid items-start gap-2 md:mx-15 md:grid-cols-[auto_1fr] md:gap-6">
                        <div>
                            <a
                                href="/articles/all"
                                className="text-2xl font-bold text-[#06A3C7] md:text-5xl"
                            >
                                Semua artikel
                            </a>
                        </div>
                        <form
                            onSubmit={handleSearch}
                            className="relative focus:outline-none"
                        >
                            <Input
                                type="text"
                                value={query}
                                onChange={(e) => setQuery(e.target.value)}
                                placeholder="Cari artikel..."
                                className="h-8 w-full rounded-full border-none bg-[#C5E9F2] pl-3 text-[10px] font-medium text-black focus-visible:ring-[2px] focus-visible:ring-[#06A3C7] md:h-12 md:pl-5 md:text-sm dark:bg-[#DDDDDD] dark:placeholder:text-black"
                            />
                            <button className="absolute top-1/2 right-4 -translate-y-1/2 cursor-pointer">
                                <Search className="h-4 w-4 text-black md:h-6 md:w-6" />
                            </button>
                        </form>
                    </section>
                    <section className="mx-5 md:mx-15">
                        <div
                            className="grid gap-3 md:grid-cols-4 md:gap-5"
                            data-aos="zoom-in"
                        >
                            {articles.data.length > 0 ? (
                                articles.data.map((article) => (
                                    <a href={`/articles/${article.slug}`}>
                                        <Card className="group cursor-pointer flex-row gap-3 overflow-hidden border-none py-0 hover:shadow-xl/30 hover:shadow-[#06A3C7] md:flex-col md:gap-2 dark:shadow-[#979797] dark:hover:shadow-xl/40 dark:hover:shadow-[#06A3C7]">
                                            <div className="relative w-[100px] flex-shrink-0 overflow-hidden bg-muted md:aspect-[3/2] md:h-auto md:w-full">
                                                <img
                                                    src={article.image_url}
                                                    className="h-full w-full object-cover"
                                                />
                                            </div>
                                            <div className="flex h-40 flex-col gap-2 py-1 pr-2 md:h-55 md:px-3 md:pt-0 md:pb-2">
                                                <div className="flex items-center gap-1">
                                                    <CalendarDays className="h-3 w-3 text-[#06A3C7] md:h-5 md:w-5" />
                                                    <p className="pt-1 text-[9px] md:text-sm">
                                                        {new Date(
                                                            article.created_at,
                                                        ).toLocaleDateString(
                                                            'id-ID',
                                                            {
                                                                day: 'numeric',
                                                                month: 'long',
                                                                year: 'numeric',
                                                            },
                                                        )}
                                                    </p>
                                                </div>
                                                <div className="flex flex-col gap-1">
                                                    <p className="text-[10px] font-semibold md:text-base">
                                                        {article.title.length >
                                                        85
                                                            ? article.title.slice(
                                                                  0,
                                                                  85,
                                                              ) + '...'
                                                            : article.title}
                                                    </p>
                                                    <div
                                                        className="text-[9px] md:text-sm"
                                                        dangerouslySetInnerHTML={{
                                                            __html:
                                                                article
                                                                    .description
                                                                    .length >
                                                                maxLength
                                                                    ? article.description.slice(
                                                                          0,
                                                                          maxLength,
                                                                      ) + '...'
                                                                    : article.description,
                                                        }}
                                                    />
                                                </div>
                                                <div className="mt-auto flex items-center gap-1">
                                                    <UserRound className="h-3 w-3 text-[#06A3C7] md:h-5 md:w-5" />
                                                    <p className="pt-1 text-[9px] md:text-sm">
                                                        Admin
                                                    </p>
                                                </div>
                                            </div>
                                        </Card>
                                    </a>
                                ))
                            ) : (
                                <p className="col-span-4 text-center text-xs text-[#969696] md:text-base">
                                    Artikel tidak ditemukan.
                                </p>
                            )}
                        </div>
                        <div
                            className="my-5 flex justify-center md:my-6"
                            data-aos="zoom-in"
                        >
                            <Stack spacing={2}>
                                <Pagination
                                    count={articles.last_page}
                                    page={articles.current_page}
                                    onChange={handlePageChange}
                                    renderItem={(item) => (
                                        <PaginationItem
                                            slots={{
                                                previous: ArrowBackIcon,
                                                next: ArrowForwardIcon,
                                            }}
                                            {...item}
                                            sx={{
                                                fontSize: {
                                                    xs: '0.75rem',
                                                    sm: '0.875rem',
                                                    md: '1rem',
                                                },
                                            }}
                                            className={`rounded-full transition-all ${
                                                item.selected
                                                    ? 'bg-gradient-to-r from-[#7DD0E2] to-[#C7FBC4] font-semibold text-black'
                                                    : 'hover:bg-[#DDDDDD]'
                                            } `}
                                        />
                                    )}
                                />
                            </Stack>
                        </div>
                    </section>
                </main>
                <Partnerships data={partnerships} />
                <Footer />
            </div>
        </>
    );
}
